/*
 * WinDom: a high level GEM library
 * Copyright (c) 1997-2006 windom authors (see AUTHORS file)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * $Source: /cvsroot/windom/windom/src/evnt_windom.c,v $
 * Module : main event handler of WinDom
 *
 * CVS info:
 *   $Author: bercegeay $
 *   $Date: 2006/04/23 16:55:37 $
 *   $Revision: 1.20 $
 */

#include <stddef.h>
#include <stdlib.h>
#include "av.h"
#include "globals.h"
#include "cookie.h"

/*
 *	management of message events.
 *
 *
 *	return 1 - stay in the event loop.
 *		   0 - leave the evnt_msg loop
 */

static 
int _do_evnt_msg( APPvar *app, short buff[8], int *evnt_res) {
	INT16 top, dum;
	INT16 x, y, w, h;
	EV_MSG *msg;
	WINDOW *win;
	
	/* Here, we support the AES4 convention for MN_SELECTED object. 
	 * This message (which can be bound to a window) does not have
	 * the window identificator in evnt.buff[3]. The way to find
	 * the targeted is different, so we process this case here */
	if( buff[0] == MN_SELECTED) {
		/* get first the menu desktop callback */
		msg = mt_EvntFind( app, NULL, MN_SELECTED);
		/* We have to find the target window 
		 * WARNING: here, buff[5,6] may be NULL if the message is generated by an "old" AES
		 * so we should check for a valid menu tree (not null)
		 */
		for( win = app->wglb.first; win; win=win->next) {
			if( win->menu.root && win->menu.root == *(OBJECT**) &buff[5]) {
				msg = mt_EvntFind( app, win, MN_SELECTED);
				break;
			}
		} 
	} else {
		/* Other messages: we verify if the message addresses a window */
		
		/* we check if this message is attached to the window buff[3] */
		win = mt_WindHandle( app, buff[3]);
		msg = mt_EvntFind( app, win, buff[0]);
		
		/* Nota: win may be NULL if the message is not adressed to a window
		 * (and as consequence, buff[3] is not the handle of a window). 
		 * We can also have a valid window handle in buff[3] althrough the
		 * message is addressed to the application (but in that case, this
		 * message is not attached to this window).
		 *
		 * At this point, we have the following cases:
		 *  -  win!=NULL && msg!=NULL  => the message is addressed to win
		 *  -  win==NULL && msg!=NULL  => the message is addressed to the application
		 *  -  win==NULL && msg==NULL  => nobody is waiting for this message
		 *  -  win!=NULL && msg==NULL  => win is not waiting for this message, we have
		 *                                to check if app is waiting for this message
		 */
		if (win && msg==NULL) {
			win = NULL;
			msg = mt_EvntFind( app, NULL, buff[0]);
		}
	}
		

	/* evnt attached to window */
	if( win && msg) {
		
		/* special cases : "sensitive" event,
			  			   emulation of some event */
		switch ( buff[0]) {
		case WM_REDRAW:
			_do_update( app, win, buff);
			break;
		case WM_TOPPED:
			/* emulate WF_BEVENT/BEVENT_WORK if the AES doesn't support it */
			if( win->status & WS_UNTOPPABLE
				&& !(app->aes4 & AES4_BEVENT)) {
				mt_wind_get( win->handle, WF_WORKXYWH, &x, &y, &w, &h, app->aes_global);
				if( app->evnt.mx >= x && app->evnt.my >= y) {
					*evnt_res |= MU_BUTTON;
					app->evnt.mbut |= 0x01;
				}
				break;
			}
			/* case of modal window */
			if( is_modal(app)) {
				mt_EvntExec( app, app->wglb.appfront, buff);
				break;
			}
			/* select menu and toolbar in background window */
			if( win->status & WS_MENU) {
				dum = mt_objc_find( win->menu.root, 0, MAX_DEPTH, app->evnt.mx, app->evnt.my, app->aes_global);
				if( dum != -1 && win->menu.root[dum].ob_type == G_TITLE ) {
					*evnt_res |= MU_BUTTON;
					app->evnt.mbut |= 0x01;
					break;
				}
			}
			if( win->status & WS_TOOLBAR) {
				dum = mt_objc_find( win->tool.root, 0, MAX_DEPTH, app->evnt.mx, app->evnt.my, app->aes_global);
				if( dum != -1 && win->tool.root[dum].ob_flags & (SELECTABLE|TOUCHEXIT) )
				{
					*evnt_res |= MU_BUTTON;
					app->evnt.mbut |= 0x01;
					break;
				}
			}

			if( win->status & WS_ICONIFY)
				std_tpd( win, buff, app);
			else {				
				mt_WindGet( app, win, WF_WORKXYWH, &x, &y, &w, &h);
				if( CONF(app)->flag & FLG_BUTMOUSE && IS_IN( app->evnt.mx, app->evnt.my, x, y, w, h)) {
					*evnt_res |= MU_BUTTON;
					app->evnt.mbut |= 0x01;
				} else 
					mt_EvntExec( app, win, buff);
			}
			break;

		case WM_BOTTOMED:
			if( !(win -> status & WS_MODAL))
				mt_WindSet( app, win, WF_BOTTOM, buff[3], 0, 0, 0);
			break;
		
		case WM_ICONIFY:
			if( win -> status & WS_MODAL && win == app->wglb.appfront)
				/* iconify a modal window => transformed to iconify_all message */
				mt_snd_msg( app, win, WM_ALLICONIFY, buff[4], buff[5], buff[6], buff[7]);
			else
				mt_EvntExec( app, win, buff);
			break;
		case WM_MOVED:
			/* update the menubar tree position */
			if( win -> status & WS_MENU &&
				!( SYSMENU(app) && win->attrib & MENUBAR) ) {
				mt_wind_get( win->handle, WF_CURRXYWH, &x, &y, &dum, &dum, app->aes_global);
				win->menu.root->ob_x += buff[4]-x;
				win->menu.root->ob_y += buff[5]-y;
			}
			/* update the toolbar tree position*/
			if( win -> status & WS_TOOLBAR ) {
				mt_wind_get( win->handle, WF_CURRXYWH, &x, &y, &dum, &dum, app->aes_global);
				win->tool.root->ob_x += buff[4]-x;
				win->tool.root->ob_y += buff[5]-y;
			}
			if( win -> status & WS_ICONIFY)
				std_mvd( win, buff, app);
			else  {
				/* TODO: transform WM_MOVED to WM_BOTTOMED ? I'm not sure to understand why... */
				if( !(app->aes4 & AES4_BOTTOM) && (app->evnt.mkstate & K_LSHIFT || app->evnt.mkstate & K_RSHIFT)) {
					mt_snd_msg( app, win, WM_BOTTOMED,0,0,0,0);
					return 1;
				}
				mt_EvntExec( app, win, buff);
			}
			break;
		case WM_FULLED:
			if( win -> status & WS_MENU &&
				!( SYSMENU(app) && win->attrib & MENUBAR) ) {
				mt_wind_get( win->handle, WF_CURRXYWH, &x, &y, &dum, &dum, app->aes_global);
				win->menu.root->ob_x += buff[4]-x;
				win->menu.root->ob_y += buff[5]-y;
			}
			if( win -> status & WS_TOOLBAR ) {
				mt_wind_get( win->handle, WF_CURRXYWH, &x, &y, &dum, &dum, app->aes_global);
				win->tool.root->ob_x += buff[4]-x;
				win->tool.root->ob_y += buff[5]-y;
			}
			mt_EvntExec( app, win, buff);
			break;
		case WM_CLOSED:
			/* -> emulate iconification <- */
			if( ((app->evnt.mkstate & (K_LSHIFT|K_RSHIFT|K_CTRL)) == (K_LSHIFT|K_RSHIFT|K_CTRL))
				&& (app->aes4 & AES4_SMALLER) == 0 ) {
				mt_give_iconifyxywh( app, &x, &y, &w, &h);
				if( app->evnt.mkstate & (K_LSHIFT|K_RSHIFT))
					mt_snd_msg( app, win, WM_ICONIFY, x, y, w, h);
				else
					mt_snd_msg( app, win, WM_ALLICONIFY, x, y, w, h);
			} else if( (app->aes4 & AES4_APPLCONTROL)
					   && ((app->evnt.mkstate & (K_LSHIFT|K_RSHIFT)) == (K_LSHIFT|K_RSHIFT))) {
				/* -> hide the application <- */
				mt_ApplControl( app, mt_AppId(app), APC_HIDE, NULL);
			} else {
				if( is_modal(app) && win != app->wglb.appfront)
					return 0;
				mt_EvntExec( app, win, buff);
			}
			break;
		
		case WM_FORM:
		case WM_TOOLBAR:
			if( *evnt_res & MU_BUTTON)
				*evnt_res &= ~MU_BUTTON;
			mt_EvntExec( app, win, buff);
			break;
			
		default:
			mt_EvntExec( app, win, buff);
		}
	} 

	/* Case of Menu binding are treated now : before
	 * the MN_SELECTED callback evaluation */
	if( buff[0] == MN_SELECTED) {
		mt_menu_bind( app, win, buff[4], buff[3]);
	}

	/* other events */
	if( !win && msg)
		mt_EvntExec( app, NULL, buff);

	/* "special" events */
	if( !msg) {
	 	switch( buff[0]) {
		case AP_BUTTON:
			/* AP_BUTTON becomes MU_BUTTON */
			*evnt_res		  &= ~MU_MESAG;
			*evnt_res		  |=  MU_BUTTON;
			app->evnt.mx   	   = buff[3];
			app->evnt.my   	   = buff[4];
			app->evnt.mbut 	   = buff[5];
			app->evnt.mkstate  = buff[6];
			app->evnt.nb_click = buff[7];
			break;
		case AP_KEYBD:
			/* AP_KEYBD becomes MU_KEYBD */
			*evnt_res		   &= ~MU_MESAG;
			*evnt_res		   |=  MU_KEYBD;
			app->evnt.mkstate 	= buff[3];
			app->evnt.keybd 	= buff[4];
			break;
		}
	} 

	/* Management of colors palet (restore the palet used by the top window) */
	/* WM_UNTOPPED = window sent to background
	 * WM_ONTOP	   = a window of another application is topped 
	 */
	if( !(CONF(app)->flag & FLG_NOPAL) && (
		   buff[ 0] == WM_UNTOPPED 
		|| buff[ 0] == WM_ONTOP 
		|| buff[ 0] == WM_TOPPED)) {
		
		mt_wind_get( 0, WF_TOP, &top, &dum, &dum, &dum, app->aes_global);
		win = mt_WindHandle( app, top);
		w_setpal(app, win);
	}

	return 0;
}

/** manage the MU_BUTTON event
 *
 *  @param app Application descriptor
 *  @param mu_button flag which indicate if the MU_BUTTON event was expected by the application
 *
 *  If the application expects a mouse button event ( mt_EvntWindom() has been called
 *  with the MU_BUTTON flag) then mu_button is 1 to indicate that the application
 *  is ready to receive WM_XBUTTON messages.
 *
 *  If the application does not expect a mouse button event ( mt_EvntWindom() has been called
 *  without the MU_BUTTON flag) then mu_button is 0 to indicate that the application
 *  doesn't want to receive WM_XBUTTON message at the moment.
 */
static void _do_evnt_mbutton( APPvar *app, int mu_button)
{
	WINDOW *mwin;
	INT16 x, y, w, h;
	EV_MSG *evmsg;
	
	/* window under the mouse pointer */
	mwin = mt_WindHandle( app, mt_wind_find( app->evnt.mx, app->evnt.my, app->aes_global));

	/* case of a modal window */
	if( is_modal(app) && mwin != app->wglb.appfront) {
		/* the modal window is topped */
		mt_snd_msg( app, app->wglb.appfront, WM_TOPPED, 0, 0, 0, 0);
		return;
	}

	evmsg = mt_EvntFind( app, mwin, WM_XBUTTON);
	
	if( mwin && app->evnt.mbut & LEFT_BUTTON) {
		/* case of an iconified window */
		if( mwin ->status & WS_ICONIFY) {
			if( app->evnt.nb_click == 2) {
				mt_WindGet( app, mwin, WF_UNICONIFY, &x, &y, &w, &h);
				mt_snd_msg( app, mwin, WM_UNICONIFY, x, y, w, h);
			}
			return;
		}
		/* menu click */
		if( mwin -> status & WS_MENU && is_menu( app, mwin)) {
#ifdef BETAMENU
	    	if( frm_menu_beta( app, mwin) != -1)
#else
	    	if( frm_menu( app, mwin) != -1)
#endif
	    		return;
	    	else {
				short buff[8] = {WM_TOPPED};
	    		mt_EvntExec( app, mwin, buff); /* top the window if nothing is selected */
			}
	    }
	    /* toolbar click */
	    if( mwin -> status & WS_TOOLBAR) {
	    	int res;

			res = frm_buttn_ev( app, mwin, OC_TOOLBAR);
	
			if( res != -1) {
				mt_snd_msg( app, mwin, WM_TOOLBAR, res, app->evnt.keybd, 0, 0);
				return;
			}
		}
		/* formular */
		if( mwin -> status & WS_FORM) {
			mt_WindGet( app, mwin, WF_WORKXYWH, &x, &y, &w, &h);
			if( IS_IN(app->evnt.mx, app->evnt.my, x, y, w, h)) {
				short buff[8];
				
				buff[0] = WM_XBUTTON;
				buff[1] = mt_AppAESapid(app);
				buff[2] = 0;
				buff[3] = app->evnt.mx;
				buff[4] = app->evnt.my;
				buff[5] = app->evnt.mbut;
				buff[6] = app->evnt.mkstate;
				buff[7] = app->evnt.nb_click;
				
				mt_EvntExec( app, mwin, buff);
				return;
			}
		}
	}

	/* other cases: send WM_XBUTTON message if mu_button is 1 */
	if( mu_button && mwin && evmsg) {
		mt_WindGet( app, mwin, WF_WORKXYWH, &x, &y, &w, &h);
		if( IS_IN(app->evnt.mx, app->evnt.my, x, y, w, h)) {
			short buff[8];

			buff[0] = WM_XBUTTON;
			buff[1] = mt_AppAESapid(app);
			buff[2] = 0;
			buff[3] = app->evnt.mx;
			buff[4] = app->evnt.my;
			buff[5] = app->evnt.mbut;
			buff[6] = app->evnt.mkstate;
			buff[7] = app->evnt.nb_click;

			mt_EvntExec( app, mwin, buff);
		}
	}
	
	/* for mouse button event attached to the application by EvntAttach( NULL) */
	if (mu_button) {
		short buff[8];

		buff[0] = WM_XBUTTON;
		buff[1] = mt_AppAESapid(app);
		buff[2] = 0;
		buff[3] = app->evnt.mx;
		buff[4] = app->evnt.my;
		buff[5] = app->evnt.mbut;
		buff[6] = app->evnt.mkstate;
		buff[7] = app->evnt.nb_click;

		mt_EvntExec( app, NULL, buff);
	}
}

static void _do_evnt_keybd( APPvar *app, int mu_keybd)
{
	INT16 dum;
	WINDOW *win;
	EV_MSG *evmsg;
	INT16 x, y, w, h;
	
	/* management of the FLG_KEYMOUSE mode */
	if( CONF(app)->flag & FLG_KEYMOUSE)
		win = mt_WindHandle( app, mt_wind_find( app->evnt.mx, app->evnt.my, app->aes_global));
	else
		win = app->wglb.front;

	evmsg = mt_EvntFind( app, win, WM_XKEYBD);

	/* case of a modal window */
	if( is_modal(app) && win != app->wglb.appfront)
		win = NULL;

	/* handle key shortcuts */
	if( !(CONF(app)->flag & FLG_NOKEYMENU)
		&& (menu_exec_cmd( app, win) || menu_exec_cmd( app, NULL)))
		;
	else if( win && win->status & WS_TOOLBAR
		     && (dum = frm_keybd_ev( app, win->tool.root)) != -1) {
		mt_objc_offset( win->tool.root, dum, &w, &h, app->aes_global);
		x = app->evnt.mx; y = app->evnt.my;
		app->evnt.mx = w + win->tool.root[dum].ob_width/2-1;
		app->evnt.my = h + win->tool.root[dum].ob_height/2-1;
		dum = frm_buttn_ev( app, win, OC_TOOLBAR);
		app->evnt.mx = x; app->evnt.my = y;
		if( dum != -1) {
			mt_snd_msg( app, win, WM_TOOLBAR, dum, app->evnt.keybd, 0, 0);
		}
	} else if ( win && (mu_keybd || win->status & WS_FORM) && evmsg) {
		short buff[8];

		buff[0] = WM_XKEYBD;
		buff[1] = mt_AppAESapid(app);
		buff[2] = 0;
		buff[3] = app->evnt.mkstate;
		buff[4] = app->evnt.keybd;
		buff[5] = 0;
		buff[6] = 0;
		buff[7] = 0;

		mt_EvntExec( app, win, buff);
	}

	/* for key events attached to the application by EvntAttach( NULL) */
	{
		short buff[8];

		buff[0] = WM_XKEYBD;
		buff[1] = mt_AppAESapid(app);
		buff[2] = 0;
		buff[3] = app->evnt.mkstate;
		buff[4] = app->evnt.keybd;
		buff[5] = 0;
		buff[6] = 0;
		buff[7] = 0;
		
		mt_EvntExec( app, NULL, buff);
	}
}

int mt_EvntWindomBuff( APPvar *app, int evntflag, short buff[8]) {
	int evnt_res;
	INT16 top, dum;
	WINDOW *win;
	
	/*  garbage: free unused userdef libraries
	 */	
	if (app->priv->udlib_garbage) {
		udlib_garbage(app);
		app->priv->udlib_garbage = 0;
	}
	
	/*
	 * the following events are treated as messages:
	 *	- desktop menu
	 *	- menu within window
	 *	- toolbar (within window)
	 *  - formulars
	 *  - key shortcut
	 *
	 *  MU_BUTTON and MU_KEYBD flag are forced to detect theses events,
	 *  by this function won't return theses event if evntflag doesn't
	 *  have these flag set.
	 */

	while (1) {
		evnt_res = mt_evnt_multi( evntflag | MU_BUTTON | MU_KEYBD | MU_MESAG,
				app->evnt.bclick, app->evnt.bmask, app->evnt.bstate,
				app->evnt.m1_flag, app->evnt.m1_x, app->evnt.m1_y, app->evnt.m1_w, app->evnt.m1_h,
				app->evnt.m2_flag, app->evnt.m2_x, app->evnt.m2_y, app->evnt.m2_w, app->evnt.m2_h,
				buff,
				app->evnt.timer,
				&app->evnt.mx, &app->evnt.my, &app->evnt.mbut, &app->evnt.mkstate,
				&app->evnt.keybd, &app->evnt.nb_click,
				app->aes_global);

	
		/* front window */
		mt_wind_get( 0, WF_TOP, &top, &dum, &dum, &dum, app->aes_global);
		app->wglb.front = mt_WindHandle( app, top);

		/* handle message events */
		if (evnt_res & MU_MESAG)
			_do_evnt_msg( app, buff, &evnt_res);

		/* handle mouse button events */
		if( evnt_res & MU_BUTTON)
			_do_evnt_mbutton(app, evntflag & MU_BUTTON);

		/* handle keyboard events */
		if( evnt_res & MU_KEYBD)
			_do_evnt_keybd(app, evntflag & MU_KEYBD);

		if( evnt_res & MU_TIMER) {
			win = app->wglb.first;
			while (1) {
				short localbuff[8] = {WM_XTIMER};
				mt_EvntExec( app, win, localbuff);
				if( !win) break;
				win = win -> next;
			}
		}

		if( evnt_res & MU_M1) {
			win = app->wglb.first;
			while (1) {
				short localbuff[8] = {WM_XM1};
				mt_EvntExec( app, win, localbuff);
				if( !win) break;
				win = win -> next;
			}
		}

		if( evnt_res & MU_M2) {
			win = app->wglb.first;
			while (1) {
				short localbuff[8] = {WM_XM2};
				mt_EvntExec( app, win, localbuff);
				if( !win) break;
				win = win -> next;
			}
		}

		if (evnt_res & evntflag)
			return (evnt_res & evntflag);
	} /* while (1) */
}

/**
 * @brief handle AES events.
 *
 * @param app application descriptor,
 * @param evntflag bit field of event to handle :
 *         - \b MU_MESAG : wait until a mesag arrives
 *         - \b MU_TIMER : wait during a fixed time
 *         - \b MU_BUTTON : wait for a mouse button event
 *         - \b MU_KEYBD : wait for a keyboard event
 *         - \b MU_M1, \b MU_M2 : wait for a motion mouse event
 * @return bit field of occured events.
 *
 * This function is the heart of WinDom. It replaces the AES
 * evnt_multi() function (the EvntMulti() is already used by the
 * Pure C AES bindings). EvntWindom() is a little bit complex:
 *  - it calls the evnt_multi() function,
 *  - it handles the color palette depending of the topped window
 *    or the desktop palette if no topped window,
 *  - it calls the good event function depending the event
 *    occured (\b MU_MESAG, \b MU_BUTTON, ...),
 *  - window menu window, window toolbar, keyboard shortcuts, specific WinDom
 *    features are handled. If needed, some AES4 special features 
 *    (iconfications, bottom windows, untoppable and modal windows) are
 *     emulated,
 *  - some new messages are eventually sent.
 * 
 * As evnt_multi(), mt_EvntWindom() can be parametrized. The
 * parametrization is performed by the variable \e app->evnt
 * (see EVNTvar structure).
 * Some events return additional informations. These
 * informations are stored in the \e app->evnt variable too. 
 *
 * \sa mt_EvntAttach(), mt_EvntAdd(), \link Event List of WinDom events\endlink, \link EVNTvar EVNTvar structure\endlink
 *
 */

int mt_EvntWindom( APPvar *app, int evntflag) {
	short buff[8];
	return mt_EvntWindomBuff( app, evntflag, buff);
}

